--[[
Module de journalisation pour FSVR
Ce module gère l'enregistrement des messages de débogage, d'information, d'avertissement et d'erreur
]]

FSVR = FSVR or {}
FSVR.Logger = {}

-- Niveaux de journalisation
FSVR.Logger.LEVEL = {
    NONE = 0,
    ERROR = 1,
    WARNING = 2,
    INFO = 3,
    DEBUG = 4
}

-- Niveau de journalisation actuel
FSVR.Logger.level = FSVR.Logger.LEVEL.INFO

-- Préfixe des messages
FSVR.Logger.prefix = "[FSVR] "

-- Fichier de journalisation
FSVR.Logger.logFile = nil
FSVR.Logger.logFileName = "FSVR.log"

-- Horodatage des messages
FSVR.Logger.useTimestamp = true

---Initialise le module de journalisation
function FSVR.Logger:init()
    -- Chargement du niveau de journalisation depuis la configuration
    if FSVR.Config then
        self.level = FSVR.Config:getNumber("general", "logLevel", self.LEVEL.INFO)
    end
    
    -- Ouverture du fichier de journalisation
    self:openLogFile()
    
    -- Message d'initialisation
    self:info("Module de journalisation initialisé (niveau: " .. self:getLevelName(self.level) .. ")")
end

---Ouvre le fichier de journalisation
function FSVR.Logger:openLogFile()
    -- Simulation de l'ouverture du fichier de journalisation
    -- Dans une implémentation réelle, cela ouvrirait un fichier en écriture
    
    self.logFile = {} -- Simulé pour le prototype
end

---Ferme le fichier de journalisation
function FSVR.Logger:closeLogFile()
    -- Simulation de la fermeture du fichier de journalisation
    -- Dans une implémentation réelle, cela fermerait le fichier
    
    self.logFile = nil
end

---Définit le niveau de journalisation
function FSVR.Logger:setLevel(level)
    if level >= self.LEVEL.NONE and level <= self.LEVEL.DEBUG then
        self.level = level
        self:info("Niveau de journalisation défini à " .. self:getLevelName(level))
        return true
    else
        self:error("Niveau de journalisation invalide: " .. tostring(level))
        return false
    end
end

---Obtient le nom d'un niveau de journalisation
function FSVR.Logger:getLevelName(level)
    if level == self.LEVEL.NONE then
        return "NONE"
    elseif level == self.LEVEL.ERROR then
        return "ERROR"
    elseif level == self.LEVEL.WARNING then
        return "WARNING"
    elseif level == self.LEVEL.INFO then
        return "INFO"
    elseif level == self.LEVEL.DEBUG then
        return "DEBUG"
    else
        return "UNKNOWN"
    end
end

---Enregistre un message
function FSVR.Logger:log(level, message)
    -- Vérification du niveau de journalisation
    if level > self.level then
        return
    end
    
    -- Construction du message
    local formattedMessage = self:formatMessage(level, message)
    
    -- Affichage du message dans la console
    self:printToConsole(formattedMessage)
    
    -- Enregistrement du message dans le fichier de journalisation
    self:writeToFile(formattedMessage)
end

---Formate un message
function FSVR.Logger:formatMessage(level, message)
    local formattedMessage = self.prefix
    
    -- Ajout de l'horodatage
    if self.useTimestamp then
        formattedMessage = formattedMessage .. self:getTimestamp() .. " "
    end
    
    -- Ajout du niveau
    formattedMessage = formattedMessage .. "[" .. self:getLevelName(level) .. "] "
    
    -- Ajout du message
    formattedMessage = formattedMessage .. tostring(message)
    
    return formattedMessage
end

---Obtient l'horodatage actuel
function FSVR.Logger:getTimestamp()
    -- Simulation de l'obtention de l'horodatage
    -- Dans une implémentation réelle, cela utiliserait os.date
    
    return os.date("%Y-%m-%d %H:%M:%S")
end

---Affiche un message dans la console
function FSVR.Logger:printToConsole(message)
    -- Simulation de l'affichage dans la console
    -- Dans une implémentation réelle, cela utiliserait print ou une fonction similaire
    
    print(message) -- Simulé pour le prototype
end

---Écrit un message dans le fichier de journalisation
function FSVR.Logger:writeToFile(message)
    -- Vérification du fichier de journalisation
    if not self.logFile then
        return
    end
    
    -- Simulation de l'écriture dans le fichier
    -- Dans une implémentation réelle, cela écrirait dans le fichier
    
    -- Pas d'implémentation spécifique pour le prototype
end

---Enregistre un message de débogage
function FSVR.Logger:debug(message)
    self:log(self.LEVEL.DEBUG, message)
end

---Enregistre un message d'information
function FSVR.Logger:info(message)
    self:log(self.LEVEL.INFO, message)
end

---Enregistre un message d'avertissement
function FSVR.Logger:warning(message)
    self:log(self.LEVEL.WARNING, message)
end

---Enregistre un message d'erreur
function FSVR.Logger:error(message)
    self:log(self.LEVEL.ERROR, message)
end

---Nettoie les ressources du module de journalisation
function FSVR.Logger:cleanup()
    -- Fermeture du fichier de journalisation
    self:closeLogFile()
end

-- Initialisation du module de journalisation
FSVR.Logger:init()

return FSVR.Logger
