--[[
Module de configuration pour FSVR
Ce module gère le chargement et la sauvegarde des paramètres de configuration
]]

FSVR = FSVR or {}
FSVR.Config = {}

-- Chemin du fichier de configuration
FSVR.Config.configPath = "config.xml"

-- Valeurs de configuration par défaut
FSVR.Config.defaultValues = {
    -- Paramètres généraux
    general = {
        enabled = true, -- Activation du mod
        debugMode = false, -- Mode de débogage
        logLevel = 1 -- Niveau de journalisation (0: aucun, 1: erreurs, 2: avertissements, 3: info, 4: debug)
    },
    
    -- Paramètres de rendu
    render = {
        resolution = 1.0, -- Facteur de résolution (1.0 = 100%)
        msaa = 4, -- Niveau de MSAA
        foveatedRendering = true, -- Rendu fovéal
        foveatedRenderingLevel = 2 -- Niveau de rendu fovéal (1: faible, 2: moyen, 3: élevé)
    },
    
    -- Paramètres d'interface utilisateur
    ui = {
        hudScale = 1.0, -- Échelle du HUD
        hudDistance = 2.0, -- Distance du HUD en mètres
        laserPointerColor = {r = 1, g = 0, b = 0, a = 0.8} -- Couleur du pointeur laser
    },
    
    -- Paramètres de contrôleur
    controllers = {
        -- Mapping des boutons des contrôleurs VR vers les actions du jeu
        mapping = {
            left = {
                trigger = "AXIS_BRAKE",
                grip = "AXIS_CLUTCH",
                primary = "INPUT_TOGGLE_LIGHTS",
                secondary = "INPUT_TOGGLE_BEACON_LIGHTS",
                menu = "INPUT_MENU_TOGGLE",
                thumbstick = {
                    x = "AXIS_MOVE_SIDE_VEHICLE",
                    y = "AXIS_MOVE_FORWARD_BACKWARD",
                    press = "INPUT_RESET"
                }
            },
            right = {
                trigger = "AXIS_ACCELERATE",
                grip = "INPUT_ATTACH",
                primary = "INPUT_IMPLEMENT_EXTRA",
                secondary = "INPUT_IMPLEMENT_EXTRA2",
                menu = "INPUT_TOGGLE_CRUISE_CONTROL",
                thumbstick = {
                    x = "AXIS_LOOK_LEFTRIGHT_VEHICLE",
                    y = "AXIS_LOOK_UPDOWN_VEHICLE",
                    press = "INPUT_TOGGLE_CAMERA"
                }
            }
        },
        
        -- Paramètres de vibration
        haptic = {
            enabled = true, -- Activation des retours haptiques
            intensity = 1.0 -- Intensité des retours haptiques
        }
    },
    
    -- Paramètres de véhicule
    vehicle = {
        steeringWheelEnabled = true, -- Activation du volant virtuel
        physicalSteeringEnabled = true, -- Activation du volant physique
        pedalEnabled = true -- Activation des pédales
    }
}

-- Valeurs de configuration actuelles
FSVR.Config.values = {}

---Initialise la configuration avec les valeurs par défaut
function FSVR.Config:init()
    -- Copie des valeurs par défaut
    self.values = self:deepCopy(self.defaultValues)
end

---Charge la configuration depuis le fichier
function FSVR.Config:load()
    -- Initialisation avec les valeurs par défaut
    self:init()
    
    -- Tentative de chargement du fichier de configuration
    local success, result = self:loadFromFile()
    
    if success then
        -- Fusion des valeurs chargées avec les valeurs par défaut
        self:mergeConfig(result)
        FSVR.Logger:info("Configuration chargée avec succès")
    else
        -- Utilisation des valeurs par défaut
        FSVR.Logger:warning("Impossible de charger la configuration, utilisation des valeurs par défaut")
        -- Sauvegarde des valeurs par défaut
        self:save()
    end
    
    return success
end

---Charge la configuration depuis le fichier
function FSVR.Config:loadFromFile()
    -- Simulation du chargement du fichier de configuration
    -- Dans une implémentation réelle, cela chargerait le fichier XML
    
    -- Vérification de l'existence du fichier
    local fileExists = false -- Simulé pour le prototype
    
    if not fileExists then
        return false, "Le fichier de configuration n'existe pas"
    end
    
    -- Chargement du fichier
    local config = {} -- Simulé pour le prototype
    
    return true, config
end

---Sauvegarde la configuration dans le fichier
function FSVR.Config:save()
    -- Simulation de la sauvegarde du fichier de configuration
    -- Dans une implémentation réelle, cela sauvegarderait le fichier XML
    
    FSVR.Logger:info("Configuration sauvegardée")
    return true
end

---Obtient une valeur de configuration
function FSVR.Config:get(section, key, defaultValue)
    -- Vérification de l'existence de la section
    if not self.values[section] then
        return defaultValue
    end
    
    -- Vérification de l'existence de la clé
    if self.values[section][key] == nil then
        return defaultValue
    end
    
    return self.values[section][key]
end

---Obtient une valeur de configuration booléenne
function FSVR.Config:getBoolean(section, key, defaultValue)
    local value = self:get(section, key, defaultValue)
    
    if type(value) == "boolean" then
        return value
    elseif type(value) == "number" then
        return value ~= 0
    elseif type(value) == "string" then
        return value:lower() == "true" or value == "1"
    else
        return defaultValue
    end
end

---Obtient une valeur de configuration numérique
function FSVR.Config:getNumber(section, key, defaultValue)
    local value = self:get(section, key, defaultValue)
    
    if type(value) == "number" then
        return value
    elseif type(value) == "string" then
        return tonumber(value) or defaultValue
    else
        return defaultValue
    end
end

---Obtient une valeur de configuration chaîne de caractères
function FSVR.Config:getString(section, key, defaultValue)
    local value = self:get(section, key, defaultValue)
    
    if type(value) == "string" then
        return value
    elseif type(value) == "number" or type(value) == "boolean" then
        return tostring(value)
    else
        return defaultValue
    end
end

---Définit une valeur de configuration
function FSVR.Config:set(section, key, value)
    -- Vérification de l'existence de la section
    if not self.values[section] then
        self.values[section] = {}
    end
    
    -- Définition de la valeur
    self.values[section][key] = value
    
    return true
end

---Réinitialise la configuration aux valeurs par défaut
function FSVR.Config:reset()
    -- Réinitialisation des valeurs
    self:init()
    
    -- Sauvegarde des valeurs par défaut
    self:save()
    
    FSVR.Logger:info("Configuration réinitialisée aux valeurs par défaut")
    return true
end

---Fusionne les valeurs de configuration
function FSVR.Config:mergeConfig(config)
    -- Parcours des sections
    for section, sectionValues in pairs(config) do
        -- Vérification de l'existence de la section
        if not self.values[section] then
            self.values[section] = {}
        end
        
        -- Parcours des clés
        for key, value in pairs(sectionValues) do
            -- Définition de la valeur
            self.values[section][key] = value
        end
    end
end

---Copie profonde d'une table
function FSVR.Config:deepCopy(original)
    local copy = {}
    
    for k, v in pairs(original) do
        if type(v) == "table" then
            copy[k] = self:deepCopy(v)
        else
            copy[k] = v
        end
    end
    
    return copy
end

-- Initialisation de la configuration
FSVR.Config:init()

return FSVR.Config
