--[[
Farming Simulator 25 VR Mod
Version: 0.1.0
]]

FSVR = {}
FSVR.MOD_NAME = g_currentModName
FSVR.MOD_DIR = g_currentModDirectory
FSVR.BASE_DIR = "."

-- Chemins vers les bibliothèques et ressources
FSVR.LIBS_DIR = FSVR.BASE_DIR .. "/libs"
FSVR.ASSETS_DIR = FSVR.BASE_DIR .. "/assets"
FSVR.SRC_DIR = FSVR.BASE_DIR .. "/src"

-- Initialisation des variables
FSVR.isVREnabled = false
FSVR.xrInstance = nil
FSVR.xrSession = nil
FSVR.headsetType = nil
FSVR.controllers = {}

-- Chargement du bridge pour le loader C#
local FSVRLoaderBridge = require("scripts/FSVRLoaderBridge")

-- Chargement des modules OpenXR
source(FSVR.SRC_DIR .. "/openxr/openxr_loader.lua")
source(FSVR.SRC_DIR .. "/openxr/openxr_instance.lua")
source(FSVR.SRC_DIR .. "/openxr/openxr_session.lua")
source(FSVR.SRC_DIR .. "/openxr/openxr_input.lua")
source(FSVR.SRC_DIR .. "/openxr/openxr_render.lua")

-- Chargement des modules d'intégration avec FS25
source(FSVR.SRC_DIR .. "/integration/camera_hook.lua")
source(FSVR.SRC_DIR .. "/integration/input_hook.lua")
source(FSVR.SRC_DIR .. "/integration/vehicle_hook.lua")
source(FSVR.SRC_DIR .. "/integration/ui_hook.lua")

-- Chargement des modules utilitaires
source(FSVR.SRC_DIR .. "/utils/config.lua")
source(FSVR.SRC_DIR .. "/utils/logger.lua")

---Initialisation du mod
function FSVR:initialize()
    FSVR.Logger:info("Initialisation du mod FSVR...")
    
    -- Chargement de la configuration
    FSVR.Config:load()
    
    -- Vérification de la compatibilité
    if not FSVR:checkCompatibility() then
        FSVR.Logger:error("Votre système n'est pas compatible avec FSVR.")
        return false
    end
    
    -- Initialisation du loader C#
    if FSVR.Config:getBoolean("general", "useNativeLoader", true) then
        FSVR.Logger:info("Utilisation du loader C# natif...")
        if not FSVRLoaderBridge:initialize() then
            FSVR.Logger:error("Échec de l'initialisation du loader C#.")
            FSVR.Logger:info("Passage en mode simulation...")
        else
            FSVR.Logger:info("Loader C# initialisé avec succès.")
            FSVR.isVREnabled = true
            return true
        end
    end
    
    -- Si le loader C# n'est pas utilisé ou a échoué, on utilise le mode simulation
    FSVR.Logger:info("Utilisation du mode simulation...")
    
    -- Initialisation d'OpenXR en mode simulation
    if not FSVR:initOpenXR() then
        FSVR.Logger:error("Échec de l'initialisation d'OpenXR.")
        return false
    end
    
    -- Mise en place des hooks pour l'intégration avec FS25
    FSVR.CameraHook:setup()
    FSVR.InputHook:setup()
    FSVR.VehicleHook:setup()
    FSVR.UIHook:setup()
    
    FSVR.Logger:info("Mod FSVR initialisé avec succès en mode simulation!")
    return true
end

---Vérification de la compatibilité du système
function FSVR:checkCompatibility()
    -- Vérification de la présence d'un runtime OpenXR
    if not FSVR.OpenXRLoader:isRuntimeAvailable() then
        FSVR.Logger:warning("Aucun runtime OpenXR détecté. Le mod fonctionnera en mode simulation.")
    end
    
    -- Vérification de la version du jeu
    if not FSVR:isGameVersionCompatible() then
        FSVR.Logger:error("Version de Farming Simulator non compatible. Version requise: 25.x.x")
        return false
    end
    
    return true
end

---Vérification de la compatibilité de la version du jeu
function FSVR:isGameVersionCompatible()
    local majorVersion = getAppVersionMajor()
    return majorVersion == 25
end

---Initialisation d'OpenXR
function FSVR:initOpenXR()
    -- Création de l'instance OpenXR
    FSVR.xrInstance = FSVR.OpenXRInstance:new()
    if not FSVR.xrInstance:initialize() then
        return false
    end
    
    -- Création de la session OpenXR
    FSVR.xrSession = FSVR.OpenXRSession:new(FSVR.xrInstance)
    if not FSVR.xrSession:initialize() then
        return false
    end
    
    -- Configuration des entrées
    if not FSVR.OpenXRInput:initialize(FSVR.xrInstance, FSVR.xrSession) then
        return false
    end
    
    -- Configuration du rendu
    if not FSVR.OpenXRRender:initialize(FSVR.xrInstance, FSVR.xrSession) then
        return false
    end
    
    FSVR.isVREnabled = true
    FSVR.headsetType = FSVR.OpenXRLoader:getHeadsetType()
    
    FSVR.Logger:info("OpenXR initialisé avec succès en mode simulation. Casque simulé: " .. tostring(FSVR.headsetType))
    return true
end

---Mise à jour du mod (appelée à chaque frame)
function FSVR:update(dt)
    if not FSVR.isVREnabled then
        return
    end
    
    -- Si le loader C# est initialisé, on l'utilise pour la mise à jour
    if FSVRLoaderBridge:isInitialized() then
        FSVRLoaderBridge:update(dt)
        return
    end
    
    -- Sinon, on utilise le mode simulation
    
    -- Mise à jour de la session OpenXR
    FSVR.xrSession:update(dt)
    
    -- Mise à jour des entrées
    FSVR.OpenXRInput:update(dt)
    
    -- Mise à jour du rendu
    FSVR.OpenXRRender:update(dt)
    
    -- Mise à jour des hooks
    FSVR.CameraHook:update(dt)
    FSVR.InputHook:update(dt)
    FSVR.VehicleHook:update(dt)
    FSVR.UIHook:update(dt)
end

---Nettoyage des ressources lors de la fermeture
function FSVR:shutdown()
    -- Si le loader C# est initialisé, on l'utilise pour la fermeture
    if FSVRLoaderBridge:isInitialized() then
        FSVRLoaderBridge:shutdown()
        FSVR.isVREnabled = false
        FSVR.Logger:info("Mod FSVR fermé avec succès via le loader C#.")
        return
    end
    
    -- Sinon, on utilise le mode simulation
    
    if FSVR.isVREnabled then
        -- Fermeture de la session OpenXR
        if FSVR.xrSession ~= nil then
            FSVR.xrSession:shutdown()
            FSVR.xrSession = nil
        end
        
        -- Fermeture de l'instance OpenXR
        if FSVR.xrInstance ~= nil then
            FSVR.xrInstance:shutdown()
            FSVR.xrInstance = nil
        end
        
        FSVR.isVREnabled = false
    end
    
    -- Nettoyage des hooks
    FSVR.CameraHook:cleanup()
    FSVR.InputHook:cleanup()
    FSVR.VehicleHook:cleanup()
    FSVR.UIHook:cleanup()
    
    FSVR.Logger:info("Mod FSVR fermé avec succès en mode simulation.")
end

-- Enregistrement des callbacks pour le cycle de vie du mod
addModEventListener(FSVR)

-- Initialisation du mod au chargement
function FSVR:loadMap(name)
    FSVR:initialize()
end

-- Nettoyage lors du déchargement
function FSVR:deleteMap()
    FSVR:shutdown()
end

-- Mise à jour à chaque frame
function FSVR:update(dt)
    FSVR:update(dt)
end

FSVR.Logger:info("Mod FSVR chargé.")
