--[[
Bibliothèque mathématique pour la VR
Cette bibliothèque fournit des fonctions mathématiques utiles pour la VR
]]

local VRMath = {}

-- Constantes mathématiques
VRMath.PI = 3.14159265358979323846
VRMath.TWO_PI = 6.28318530717958647692
VRMath.HALF_PI = 1.57079632679489661923
VRMath.DEG_TO_RAD = 0.01745329251994329576
VRMath.RAD_TO_DEG = 57.2957795130823208768

-- Fonctions vectorielles 3D
VRMath.Vector3 = {}

-- Crée un nouveau vecteur 3D
function VRMath.Vector3.new(x, y, z)
    return {x = x or 0, y = y or 0, z = z or 0}
end

-- Addition de vecteurs
function VRMath.Vector3.add(a, b)
    return {x = a.x + b.x, y = a.y + b.y, z = a.z + b.z}
end

-- Soustraction de vecteurs
function VRMath.Vector3.subtract(a, b)
    return {x = a.x - b.x, y = a.y - b.y, z = a.z - b.z}
end

-- Multiplication par un scalaire
function VRMath.Vector3.multiply(v, s)
    return {x = v.x * s, y = v.y * s, z = v.z * s}
end

-- Division par un scalaire
function VRMath.Vector3.divide(v, s)
    if s == 0 then
        error("Division par zéro")
    end
    return {x = v.x / s, y = v.y / s, z = v.z / s}
end

-- Produit scalaire
function VRMath.Vector3.dot(a, b)
    return a.x * b.x + a.y * b.y + a.z * b.z
end

-- Produit vectoriel
function VRMath.Vector3.cross(a, b)
    return {
        x = a.y * b.z - a.z * b.y,
        y = a.z * b.x - a.x * b.z,
        z = a.x * b.y - a.y * b.x
    }
end

-- Norme d'un vecteur
function VRMath.Vector3.length(v)
    return math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z)
end

-- Normalisation d'un vecteur
function VRMath.Vector3.normalize(v)
    local len = VRMath.Vector3.length(v)
    if len == 0 then
        return {x = 0, y = 0, z = 0}
    end
    return {x = v.x / len, y = v.y / len, z = v.z / len}
end

-- Distance entre deux vecteurs
function VRMath.Vector3.distance(a, b)
    return VRMath.Vector3.length(VRMath.Vector3.subtract(a, b))
end

-- Fonctions quaternions
VRMath.Quaternion = {}

-- Crée un nouveau quaternion
function VRMath.Quaternion.new(x, y, z, w)
    return {x = x or 0, y = y or 0, z = z or 0, w = w or 1}
end

-- Quaternion identité
function VRMath.Quaternion.identity()
    return {x = 0, y = 0, z = 0, w = 1}
end

-- Multiplication de quaternions
function VRMath.Quaternion.multiply(a, b)
    return {
        x = a.w * b.x + a.x * b.w + a.y * b.z - a.z * b.y,
        y = a.w * b.y - a.x * b.z + a.y * b.w + a.z * b.x,
        z = a.w * b.z + a.x * b.y - a.y * b.x + a.z * b.w,
        w = a.w * b.w - a.x * b.x - a.y * b.y - a.z * b.z
    }
end

-- Conjugué d'un quaternion
function VRMath.Quaternion.conjugate(q)
    return {x = -q.x, y = -q.y, z = -q.z, w = q.w}
end

-- Norme d'un quaternion
function VRMath.Quaternion.length(q)
    return math.sqrt(q.x * q.x + q.y * q.y + q.z * q.z + q.w * q.w)
end

-- Normalisation d'un quaternion
function VRMath.Quaternion.normalize(q)
    local len = VRMath.Quaternion.length(q)
    if len == 0 then
        return {x = 0, y = 0, z = 0, w = 1}
    end
    return {x = q.x / len, y = q.y / len, z = q.z / len, w = q.w / len}
end

-- Création d'un quaternion à partir d'un axe et d'un angle
function VRMath.Quaternion.fromAxisAngle(axis, angle)
    local halfAngle = angle * 0.5
    local s = math.sin(halfAngle)
    local normalizedAxis = VRMath.Vector3.normalize(axis)
    
    return {
        x = normalizedAxis.x * s,
        y = normalizedAxis.y * s,
        z = normalizedAxis.z * s,
        w = math.cos(halfAngle)
    }
end

-- Création d'un quaternion à partir d'angles d'Euler (en radians)
function VRMath.Quaternion.fromEuler(x, y, z)
    local cx = math.cos(x * 0.5)
    local sx = math.sin(x * 0.5)
    local cy = math.cos(y * 0.5)
    local sy = math.sin(y * 0.5)
    local cz = math.cos(z * 0.5)
    local sz = math.sin(z * 0.5)
    
    return {
        x = sx * cy * cz - cx * sy * sz,
        y = cx * sy * cz + sx * cy * sz,
        z = cx * cy * sz - sx * sy * cz,
        w = cx * cy * cz + sx * sy * sz
    }
end

-- Conversion d'un quaternion en angles d'Euler (en radians)
function VRMath.Quaternion.toEuler(q)
    local x, y, z
    
    -- Roll (x-axis rotation)
    local sinr_cosp = 2 * (q.w * q.x + q.y * q.z)
    local cosr_cosp = 1 - 2 * (q.x * q.x + q.y * q.y)
    x = math.atan2(sinr_cosp, cosr_cosp)
    
    -- Pitch (y-axis rotation)
    local sinp = 2 * (q.w * q.y - q.z * q.x)
    if math.abs(sinp) >= 1 then
        y = math.pi / 2 * (sinp > 0 and 1 or -1)
    else
        y = math.asin(sinp)
    end
    
    -- Yaw (z-axis rotation)
    local siny_cosp = 2 * (q.w * q.z + q.x * q.y)
    local cosy_cosp = 1 - 2 * (q.y * q.y + q.z * q.z)
    z = math.atan2(siny_cosp, cosy_cosp)
    
    return {x = x, y = y, z = z}
end

-- Rotation d'un vecteur par un quaternion
function VRMath.Quaternion.rotateVector(q, v)
    local qv = {x = v.x, y = v.y, z = v.z, w = 0}
    local qc = VRMath.Quaternion.conjugate(q)
    local qr = VRMath.Quaternion.multiply(VRMath.Quaternion.multiply(q, qv), qc)
    
    return {x = qr.x, y = qr.y, z = qr.z}
end

-- Interpolation linéaire entre deux quaternions
function VRMath.Quaternion.lerp(a, b, t)
    t = math.max(0, math.min(1, t))
    
    return VRMath.Quaternion.normalize({
        x = a.x + (b.x - a.x) * t,
        y = a.y + (b.y - a.y) * t,
        z = a.z + (b.z - a.z) * t,
        w = a.w + (b.w - a.w) * t
    })
end

-- Interpolation sphérique entre deux quaternions
function VRMath.Quaternion.slerp(a, b, t)
    t = math.max(0, math.min(1, t))
    
    local dot = a.x * b.x + a.y * b.y + a.z * b.z + a.w * b.w
    
    -- Si les quaternions sont très proches, utiliser lerp
    if dot > 0.9995 then
        return VRMath.Quaternion.lerp(a, b, t)
    end
    
    -- Assurer le chemin le plus court
    if dot < 0 then
        b = {x = -b.x, y = -b.y, z = -b.z, w = -b.w}
        dot = -dot
    end
    
    dot = math.min(1, math.max(-1, dot))
    
    local theta0 = math.acos(dot)
    local theta = theta0 * t
    
    local s0 = math.cos(theta) - dot * math.sin(theta) / math.sin(theta0)
    local s1 = math.sin(theta) / math.sin(theta0)
    
    return {
        x = a.x * s0 + b.x * s1,
        y = a.y * s0 + b.y * s1,
        z = a.z * s0 + b.z * s1,
        w = a.w * s0 + b.w * s1
    }
end

-- Fonctions matricielles 4x4
VRMath.Matrix4 = {}

-- Crée une nouvelle matrice 4x4 (identité par défaut)
function VRMath.Matrix4.new()
    return {
        1, 0, 0, 0,
        0, 1, 0, 0,
        0, 0, 1, 0,
        0, 0, 0, 1
    }
end

-- Matrice identité
function VRMath.Matrix4.identity()
    return {
        1, 0, 0, 0,
        0, 1, 0, 0,
        0, 0, 1, 0,
        0, 0, 0, 1
    }
end

-- Multiplication de matrices
function VRMath.Matrix4.multiply(a, b)
    local result = {}
    
    for i = 0, 3 do
        for j = 0, 3 do
            local sum = 0
            for k = 0, 3 do
                sum = sum + a[i * 4 + k + 1] * b[k * 4 + j + 1]
            end
            result[i * 4 + j + 1] = sum
        end
    end
    
    return result
end

-- Création d'une matrice de translation
function VRMath.Matrix4.translation(x, y, z)
    return {
        1, 0, 0, 0,
        0, 1, 0, 0,
        0, 0, 1, 0,
        x, y, z, 1
    }
end

-- Création d'une matrice d'échelle
function VRMath.Matrix4.scaling(x, y, z)
    return {
        x, 0, 0, 0,
        0, y, 0, 0,
        0, 0, z, 0,
        0, 0, 0, 1
    }
end

-- Création d'une matrice de rotation autour de l'axe X
function VRMath.Matrix4.rotationX(angle)
    local c = math.cos(angle)
    local s = math.sin(angle)
    
    return {
        1, 0, 0, 0,
        0, c, s, 0,
        0, -s, c, 0,
        0, 0, 0, 1
    }
end

-- Création d'une matrice de rotation autour de l'axe Y
function VRMath.Matrix4.rotationY(angle)
    local c = math.cos(angle)
    local s = math.sin(angle)
    
    return {
        c, 0, -s, 0,
        0, 1, 0, 0,
        s, 0, c, 0,
        0, 0, 0, 1
    }
end

-- Création d'une matrice de rotation autour de l'axe Z
function VRMath.Matrix4.rotationZ(angle)
    local c = math.cos(angle)
    local s = math.sin(angle)
    
    return {
        c, s, 0, 0,
        -s, c, 0, 0,
        0, 0, 1, 0,
        0, 0, 0, 1
    }
end

-- Création d'une matrice de rotation à partir d'un quaternion
function VRMath.Matrix4.fromQuaternion(q)
    local x, y, z, w = q.x, q.y, q.z, q.w
    local xx, xy, xz = x * x, x * y, x * z
    local yy, yz, zz = y * y, y * z, z * z
    local wx, wy, wz = w * x, w * y, w * z
    
    return {
        1 - 2 * (yy + zz), 2 * (xy - wz), 2 * (xz + wy), 0,
        2 * (xy + wz), 1 - 2 * (xx + zz), 2 * (yz - wx), 0,
        2 * (xz - wy), 2 * (yz + wx), 1 - 2 * (xx + yy), 0,
        0, 0, 0, 1
    }
end

-- Création d'une matrice de vue (lookAt)
function VRMath.Matrix4.lookAt(eye, target, up)
    local z = VRMath.Vector3.normalize(VRMath.Vector3.subtract(eye, target))
    local x = VRMath.Vector3.normalize(VRMath.Vector3.cross(up, z))
    local y = VRMath.Vector3.cross(z, x)
    
    return {
        x.x, y.x, z.x, 0,
        x.y, y.y, z.y, 0,
        x.z, y.z, z.z, 0,
        -VRMath.Vector3.dot(x, eye), -VRMath.Vector3.dot(y, eye), -VRMath.Vector3.dot(z, eye), 1
    }
end

-- Création d'une matrice de projection perspective
function VRMath.Matrix4.perspective(fovY, aspect, near, far)
    local f = 1.0 / math.tan(fovY * 0.5)
    local nf = 1.0 / (near - far)
    
    return {
        f / aspect, 0, 0, 0,
        0, f, 0, 0,
        0, 0, (far + near) * nf, -1,
        0, 0, 2 * far * near * nf, 0
    }
end

-- Fonctions d'interpolation
VRMath.Interpolation = {}

-- Interpolation linéaire
function VRMath.Interpolation.lerp(a, b, t)
    return a + (b - a) * t
end

-- Interpolation avec easing (ease-in-out)
function VRMath.Interpolation.easeInOut(t)
    return t * t * (3 - 2 * t)
end

-- Interpolation avec rebond (bounce)
function VRMath.Interpolation.bounce(t)
    local a = 4.0 / 11.0
    local b = 8.0 / 11.0
    local c = 9.0 / 10.0
    
    local ca = 4356.0 / 361.0
    local cb = 35442.0 / 1805.0
    local cc = 16061.0 / 1805.0
    
    local t2 = t * t
    
    if t < a then
        return 7.5625 * t2
    elseif t < b then
        t = t - a
        return 7.5625 * t2 + c
    elseif t < 0.9 then
        t = t - b
        return 7.5625 * t2 + 0.9375
    else
        t = t - 0.95
        return 7.5625 * t2 + 0.984375
    end
end

return VRMath
